<?php

namespace Cocode\DBModel\Tests;

use Cocode\DBModel\DBModelTrait;
use Nette\Caching\Storages\MemoryStorage;
use Nette\Database\Connection;
use Nette\Database\Context;
use Nette\Database\Conventions\DiscoveredConventions;
use Nette\Database\IRow;
use Nette\Database\Structure;
use Nette\Utils\ArrayHash;
use PHPUnit\Framework\TestCase;

class DBModelTraitTestMock
{

    use DBModelTrait;

    public function __construct(
        Context $database,
        string $tableName
    )
    {
        $this->database = $database;
        $this->tableName = $tableName;
    }
}

class DBModelTraitTest extends TestCase
{
    /**
     * @var DBModelTraitTestMock
     */
    protected static $model;

    /**
     * @var Context
     */
    protected static $context;

    /**
     * @var string
     */
    protected static $dbStructure;

    /**
     * @var string
     */
    protected static $dbData;

    const
        CATALOG_TABLE = 'catalog',
        ITEMS_TABLE = 'items',
        VARIANTS_TABLE = 'variants';

    protected static function rowsToArray(array $rows): array
    {
        $result = [];
        foreach ($rows as $key => $row) {
            $result[$key] = $row->toArray();
        }
        return $result;
    }

    public static function setUpBeforeClass(): void
    {
        $connection = new Connection(
            'mysql:host=localhost;dbname=cocobox',
            'root',
            'cocobox',
            [
                'lazy' => 'yes'
            ]
        );
        $storage = new MemoryStorage();
        $structure = new Structure($connection, $storage);
        $conventions = new DiscoveredConventions($structure);
        $context = new Context(
            $connection,
            $structure,
            $conventions,
            $storage
        );
        self::$model = new DBModelTraitTestMock($context, self::CATALOG_TABLE);
        self::$context = $context;
        self::$dbStructure = file_get_contents(__DIR__ . '/sql/structure.sql');
        self::$dbData = file_get_contents(__DIR__ . '/sql/data.sql');
    }

    public static function tearDownAfterClass(): void
    {
        self::$model = null;
        self::$context = null;
    }

    public function setUp(): void
    {
        parent::setUp();
        $database = self::$context;
        $database->query(self::$dbStructure);
        $database->query(self::$dbData);
    }

    public function tearDown(): void
    {
        parent::tearDown();
        self::$context->query('
            SET FOREIGN_KEY_CHECKS = 0;
            DROP TABLE IF EXISTS ' . self::CATALOG_TABLE . ';
            DROP TABLE IF EXISTS ' . self::ITEMS_TABLE . ';
            DROP TABLE IF EXISTS ' . self::VARIANTS_TABLE . ';
            SET FOREIGN_KEY_CHECKS = 1;
        ');
    }

    public function testFormValuesToRow()
    {
        $values = ArrayHash::from(
            [
                'id' => 2,
                'title' => 'catalog updated',
                'last_change' => new \DateTime('2019-04-11 18:40:59')
            ]
        );
        $model = self::$model;
        $model->formValuesToRow($values, self::CATALOG_TABLE);
        $row = $model->getRecord(2, self::CATALOG_TABLE);
        $this->assertEquals(
            [
                'id' => 2,
                'title' => 'catalog updated',
                'last_change' => new \DateTime('2019-04-11 18:40:59')
            ],
            $row->toArray(),
            'Row data is not same'
        );
    }

    public function testGetRecordWhere()
    {
        $model = self::$model;
        $record = $model->getRecordWhere('id > ?', 1, self::CATALOG_TABLE);
        $this->assertTrue($record instanceof IRow, '$record is not an IRow instance');
        $this->assertEquals([
            'id' => 2,
            'title' => 'catalog 2',
            'last_change' => new \DateTime('2019-04-11 18:40:59')
        ], $record->toArray(), 'Row data not equals blueprint array.');

        $record = $model->getRecordWhere('id > ?', 1234, self::CATALOG_TABLE);
        $this->assertTrue($record === null, '$record is not null');
    }

    public function testGetRecord()
    {
        $model = self::$model;
        $record = $model->getRecord(1, self::CATALOG_TABLE);
        $this->assertTrue($record instanceof IRow, '$record is not an IRow instance');
        $this->assertEquals([
            'id' => 1,
            'title' => 'catalog 1',
            'last_change' => new \DateTime('2019-04-11 18:40:51')
        ], $record->toArray(), 'Row data not equals blueprint array.');

        $record = $model->getRecord(2, self::CATALOG_TABLE, 'title');
        $this->assertTrue($record instanceof IRow, '$record is not an IRow instance');
        $this->assertEquals([
            'title' => 'catalog 2'
        ], $record->toArray(), 'Row data not equals blueprint array.');

        $record = $model->getRecord(1234, self::CATALOG_TABLE);
        $this->assertTrue($record === null, '$record is not null');
    }

    public function testCommitDBTransaction()
    {
        $model = self::$model;
        self::$context->beginTransaction();
        self::$context->table(self::CATALOG_TABLE)
            ->where('NOT(id IS NULL)')
            ->delete();
        $model->commitDBTransaction();

        $record = $model->getRecord(1, self::CATALOG_TABLE);
        $this->assertTrue($record === null, 'Transaction was not committed');
    }

    public function testTableExists()
    {
        $model = self::$model;
        $this->assertTrue($model->tableExists(self::CATALOG_TABLE), 'Table was not found');
        $this->assertTrue($model->tableExists('foo') === false, 'Table was found');
    }

    public function testGetPairsFor()
    {
        $model = self::$model;
        $pairs = $model->getPairsFor(self::CATALOG_TABLE, 'id', 1, 'title');
        $this->assertEquals(
            [
                1 => 'catalog 1',
            ],
            $pairs,
            'Pairs are not equal'
        );
    }

    public function testGetPairs()
    {
        $model = self::$model;
        $pairs = $model->getPairs(self::CATALOG_TABLE, 'id', 'title');
        $this->assertEquals(
            [
                1 => 'catalog 1',
                2 => 'catalog 2'
            ],
            $pairs,
            'Pairs are not equal'
        );
    }

    public function testGetTableName()
    {
        $this->assertEquals(
            self::CATALOG_TABLE,
            self::$model->getTableName(),
            'Table name do not match.'
        );
    }

    public function testDeleteRecord()
    {
        $model = self::$model;
        $this->assertTrue(
            $model->deleteRecord(1),
            'Record was not stated as deleted'
        );
        $this->assertTrue(
            $model->deleteRecord(1, self::ITEMS_TABLE),
            'Record was not stated as deleted'
        );
        $this->assertTrue(
            self::$context->table(self::CATALOG_TABLE)->get(1) === false,
            'Record was not deleted'
        );
        $this->assertTrue(
            self::$context->table(self::ITEMS_TABLE)->get(1) === false,
            'Record was not deleted'
        );
    }

    /*
    public function testGetEnsuredUUID()
    {
        non-testable
    }
    */

    public function testGetAll()
    {
        $model = self::$model;
        $expected = self::$context->table(self::CATALOG_TABLE)->fetchAll();
        $actual = $model->getAll(self::CATALOG_TABLE);

        $this->assertEquals(
            self::rowsToArray($expected),
            self::rowsToArray($actual),
            'Result sets does not match'
        );
        $this->assertEquals(
            self::rowsToArray(self::$context->table(self::CATALOG_TABLE)->where('id = ?', 1)->fetchAll()),
            self::rowsToArray($model->getAll(self::CATALOG_TABLE, ['id' => 1])),
            'Result sets does not match'
        );
    }

    public function testBeginDBTransaction()
    {
        $model = self::$model;
        $model->beginDBTransaction();
        self::$context->table(self::CATALOG_TABLE)->where('id = ?', 1)->delete();
        $this->expectException(
            \PDOException::class
        );
        self::$context->beginTransaction();
        self::$context->commit();
        $this->assertTrue(
            self::$context->table(self::CATALOG_TABLE)->get(1) === false,
            'Transaction was not committed'
        );

        $this->assertTrue($model->beginDBTransaction());
        $this->assertTrue(!$model->beginDBTransaction());
        self::$context->rollBack();
    }

    public function testRollbackDBTransaction()
    {
        $model = self::$model;
        self::$context->rollBack();
        self::$context->beginTransaction();
        self::$context->table(self::CATALOG_TABLE)->where('id = ?', 1)->delete();
        $model->rollbackDBTransaction();
        $this->assertTrue(
            self::$context->table(self::CATALOG_TABLE)->get(1) instanceof IRow,
            'Transaction was committed'
        );
    }

    public function testInsertUpdateRecord()
    {
        $model = self::$model;
        $insert = [
            'id' => 4,
            'title' => 'inserted',
            'last_change' => new \DateTime('2019-04-11 18:40:59')
        ];
        $insert2 = [
            'id' => null,
            'title' => 'inserted',
            'last_change' => new \DateTime('2019-04-11 18:40:59')
        ];
        $update = [
            'id' => 2,
            'title' => 'updated',
            'last_change' => new \DateTime('2019-04-11 18:40:59')
        ];

        $insertedId = $model->insertUpdateRecord($insert, self::CATALOG_TABLE, $insert['id']);
        $this->assertEquals(4, $insertedId, 'Ids is not same');
        $this->assertEquals($insert, $model->getRecord(4)->toArray(), 'Data is not equal');
        $insertedId = $model->insertUpdateRecord($insert2, self::CATALOG_TABLE);
        $this->assertEquals(5, $insertedId, 'Ids is not same');
        $this->assertEquals(array_merge($insert, ['id' => $insertedId]), $model->getRecord($insertedId)->toArray(), 'Data is not equal');
        $updatedId = $model->insertUpdateRecord($update, self::CATALOG_TABLE, $update['id']);
        $this->assertEquals($update['id'], $updatedId, 'Ids is not same');
        $this->assertEquals($update, $model->getRecord($update['id'])->toArray(), 'Data is not equal');
    }

    public function testGetPairsWhere()
    {
        $model = self::$model;
        $pairs = $model->getPairsWhere('id = ?', 1, 'id', 'title', self::CATALOG_TABLE);
        $this->assertEquals(
            [
                1 => 'catalog 1',
            ],
            $pairs,
            'Pairs are not equal'
        );
    }

    public function testGetRecordsWhere()
    {
        $model = self::$model;
        $rows = $model->getRecordsWhere('id > ?', 1, self::CATALOG_TABLE);
        $this->assertEquals(
            [
                2 => [
                    'id' => 2,
                    'title' => 'catalog 2',
                    'last_change' => new \DateTime('2019-04-11 18:40:59')
                ]
            ],
            $this->rowsToArray($rows),
            'Rows should be same'
        );
    }
}
