<?php

namespace Cocode\DBModel;

use Hidehalo\Nanoid\Client;
use Nette\Database\Context;
use Nette\Database\ActiveRow;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

/**
 * Trait DBModelTrait - Basic nette database wrapper functions
 * @package App\Model\Traits
 */
trait DBModelTrait
{

    /** @var Context */
    public $database;

    /** @var string */
    public $tableName;

    /**
     * Returns row from selected table or false if it does not exist
     *
     * @param integer $id
     * @param string|null $tableName
     * @param string $select
     * @return ActiveRow|null
     */
    public function getRecord(int $id, string $tableName = null, string $select = '*'): ?ActiveRow
    {
        $row = $this->database->table((!is_null($tableName)) ? $tableName : $this->tableName)
            ->select($select)
            ->limit(1)
            ->get($id);
        return (!empty($row)) ? $row : null;
    }

    /**
     * Returns row from selected table where cond suits passed parameter
     * @param string $cond
     * @param mixed $parameter
     * @param string $tableName
     * @return ActiveRow|null or false
     * @example path getRecordWhere('id >= ?', 123, 'users');
     */
    public function getRecordWhere(string $cond, $parameter, string $tableName = null): ?ActiveRow
    {
        $row = $this->database->table((!is_null($tableName)) ? $tableName : $this->tableName)
            ->where($cond, $parameter)
            ->limit(1)
            ->fetch();
        return (!empty($row)) ? $row : null;
    }

    /**
     * Returns selection of rows from table where cond suits passed parameter
     * @param string $cond
     * @param mixed $parameter
     * @param string $tableName
     * @return array|null or false
     * @example path getRecordsWhere('id >= ?', 123, 'users');
     */
    public function getRecordsWhere(string $cond, $parameter, string $tableName = null): ?array
    {
        $rows = $this->database->table((!is_null($tableName)) ? $tableName : $this->tableName)
            ->where($cond, $parameter)->fetchAll();
        return (!empty($rows)) ? $rows : null;
    }

    /**
     * Deletes record from table
     * @param int $id
     * @param string $tableName
     * @return boolean true on success
     */
    public function deleteRecord(int $id, string $tableName = null): bool
    {
        $record = $this->getRecord($id, ($tableName != null) ? $tableName : $this->tableName);
        if ($record != null) {
            try {
                $record->delete();
                return true;
            } catch (\PDOException $e) {
                Debugger::log('Nepodařilo se smazat záznam. Chyba: ' . $e->getMessage(), Debugger::ERROR);
                return false;
            }
        } else {
            Debugger::log('Nenalezen záznam pro smazání.');
            return false;
        }
    }

    /**
     * Inserts or updates data of row in database
     * @param array|ArrayHash $data
     * @param string $tableName
     * @param int $id
     * @return int id of inserted/updated row
     */
    public function insertUpdateRecord($data, string $tableName = null, int $id = null): ?int
    {
        $table = ($tableName !== null) ? $tableName : $this->tableName;
        $primaryKey = $this->database->table($table)->getPrimary();
        if ($id !== null && $id > 0) {
            $rec = $this->getRecord($id, $table);
            if ($rec instanceof ActiveRow) {
                $rec->update($data);
                return $id;
            }
        }
        if (empty($id) || $id === 0 || empty($rec)) {
            $result = $this->database->table($table)->insert($data);
            if (empty($result)) {
                $result = $this->database->getInsertId();
            }
            if (empty($result)) {
                $result = $this->database->table($table)->order($primaryKey . ' DESC')->limit(1)->fetch();
            }
            return (($result instanceof ActiveRow) ? $result[$primaryKey] : ((empty($result)) ? null : (int)$result));
        }
    }

    /**
     * Returns all rows from table
     * @param string $tableName
     * @param array $where ['fieldName'=>'fieldValue'] (fieldname == fieldvalue)
     * @return array
     */
    public function getAll(string $tableName = null, array $where = []): ?array
    {
        $tableName = ($tableName == null) ? $this->tableName : $tableName;
        $rows = (empty($where)) ? $this->database->table($tableName)->fetchAll() : $this->database->table($tableName)->where($where)->fetchAll();
        return (!empty($rows)) ? $rows : null;
    }

    /**
     * Returns name of main table
     * @return string
     */
    public function getTableName(): string
    {
        return $this->tableName;
    }

    /**
     * Checks if table exists in current database
     * @param string $tableName
     * @return boolean
     */
    public function tableExists(string $tableName): bool
    {
        $tables = $this->database->getStructure()->getTables();
        foreach ($tables as $table) {
            if ($table['name'] === $tableName) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns array of pairs $keyField=>$valueField
     * @param string $tableName
     * @param string $keyField
     * @param string $valueField
     * @return array
     */
    public function getPairs(string $tableName, string $keyField, string $valueField): array
    {
        return $this->database->table($tableName)->fetchPairs($keyField, $valueField);
    }

    /**
     * Returns array of row Id=>$valueField values where $keyFieldName = $keyValue
     * @param string $tableName
     * @param string $keyFieldName
     * @param mixed $keyValue
     * @param string $valueField
     * @return array
     */
    public function getPairsFor(string $tableName, string $keyFieldName, $keyValue, string $valueField): array
    {
        return $this->database->table($tableName)->where($keyFieldName . ' = ?', $keyValue)->fetchPairs('id', $valueField);
    }

    /**
     * Returs array of $keyField=>$valueField from table $tableName where $cond suits $parameter
     * @param string $cond 'id => ?'
     * @param mixed $parameter 5
     * @param string $keyField 'username'
     * @param string $valueField 'password'
     * @param string $tableName 'users'
     * @return array
     */
    public function getPairsWhere(string $cond, $parameter, string $keyField, string $valueField, string $tableName = null): array
    {
        return $this->database->table(($tableName !== null) ? $tableName : $this->tableName)->where($cond, $parameter)->fetchPairs($keyField, $valueField);
    }

    /**
     * @param ArrayHash $values
     * @param string $tableName
     * @return int|null
     */
    public function formValuesToRow(ArrayHash $values, string $tableName): ?int
    {
        return $this->insertUpdateRecord($values, $tableName, (property_exists($values, 'id') ? $values['id'] : null));
    }

    /**
     * Generates nanoid and ensures uniqueness in passed table and field
     * @param string $fieldName
     * @param string $tableName
     * @return string
     */
    public function getEnsuredUUID(string $fieldName, string $tableName): string
    {
        $client = new Client();
        $uuid = $client->generateId();
        while ($this->database->table($tableName)->where($fieldName . ' = ?', $uuid)->count('*') > 0) {
            $uuid = $client->generateId();
        }
        return $uuid;
    }

    /**
     * Begins db transaction
     *
     * @return bool
     */
    public function beginDBTransaction(): bool
    {
        if (!$this->isInTrasaction()) {
            $this->database->beginTransaction();
            return true;
        }
        return false;
    }

    public function isInTrasaction(): bool
    {
        return $this->database->getConnection()->getPdo()->inTransaction();
    }

    /**
     * Commits db transaction
     *
     * @return bool
     */
    public function commitDBTransaction(): bool
    {
        if ($this->isInTrasaction()) {
            $this->database->commit();
            return true;
        }
        return false;
    }

    /**
     * Rolls back db transaction
     *
     * @return bool
     */
    public function rollbackDBTransaction(): bool
    {
        if ($this->isInTrasaction()) {
            $this->database->rollBack();
            return true;
        }
        return false;
    }

}
