<?php

namespace Cocode\DBModel;

use Nette\Database\Table\ActiveRow;

trait DBModelAssignsTrait
{
    use DBModelTrait;

    /**
     * Creates new assignment record in target database
     * @param string $tableName
     * @param array $pairing ['keyName1'=>$value1, 'keyName2'=>$value2]
     * @param array $aditionalFields ['fieldName'=>value]
     * @return boolean false if exists
     */
    public function assign(string $tableName, array $pairing, array $aditionalFields = []): bool
    {
        $fieldNames = array_keys($pairing);
        $ids = array_values($pairing);
        $record = $this->database->table($tableName)->where($fieldNames[0] . ' = ? AND ' . $fieldNames[1] . ' = ?', $ids[0], $ids[1]);
        if (!empty($aditionalFields)) {
            foreach ($aditionalFields as $fieldName => $fieldValue) {
                $record->where($fieldName . ' = ?', $fieldValue);
            }
        }
        $record = $record->limit(1)->fetch();
        if (empty($record)) {
            $insert = (!empty($aditionalFields)) ? array_merge($pairing, $aditionalFields) : $pairing;
            $this->database->table($tableName)->insert($insert);
            return true;
        }
        return false;
    }

    /**
     * Deletes assignment record in target database if exists
     * @param string $tableName
     * @param array $pairing ['keyName1'=>$value1, 'keyName2'=>$value2]
     * @return boolean false if doesnt exists true on success
     */
    public function unassign(string $tableName, array $pairing): bool
    {
        $fieldNames = array_keys($pairing);
        $ids = array_values($pairing);
        $record = $this->database->table($tableName)->where($fieldNames[0] . ' = ? AND ' . $fieldNames[1] . ' = ?', $ids[0], $ids[1])->limit(1)->fetch();
        if (!empty($record)) {
            $record->delete();
            return true;
        }
        return false;
    }

    /**
     * Returns Selection of records assigned to passed ActiveRow through pairing table
     * @param ActiveRow $record
     * @param string $pairingTableName - name of table with pairing records
     * @param string $throughKey - target table key field name in pairing records
     * @param string $targetTableName - name of table containing desired records
     * @return array|null
     */
    public function getAssigned(ActiveRow $record, string $pairingTableName, string $throughKey, string $targetTableName): ?array
    {
        $assignedIds = array_values($record->related($pairingTableName)->fetchPairs('id', $throughKey));
        return (!empty($assignedIds)) ? $this->database->table($targetTableName)->where('id IN(?)', $assignedIds)->fetchAll() : null;
    }

    /**
     * Returns Selection of records not assigned to passed ActiveRow through pairing table
     * @param ActiveRow $record
     * @param string $pairingTableName - name of table with pairing records
     * @param string $throughKey - target table key field name in pairing records
     * @param string $targetTableName - name of table containing desired records
     * @return array|null | boolean false
     */
    public function getUnassigned(ActiveRow $record, string $pairingTableName, string $throughKey, string $targetTableName): ?array
    {
        $assignedIds = array_values($record->related($pairingTableName)->fetchPairs('id', $throughKey));
        return (!empty($assignedIds)) ? $this->database->table($targetTableName)->where('NOT(id IN(?))', $assignedIds)->fetchAll() : $this->getAll($targetTableName);
    }

    /**
     * Updates current assignments in pairing table
     * @param string $pairingTableName - name of pairing table
     * @param string $assignmentKeyName - name of key field used for reference with target table
     * @param array $assignmentKeyValues - array of target key values
     * @param string $keyName - name of key field used for reference with source table
     * @param int $keyValue - value for source key field
     */
    public function updateAssignments(string $pairingTableName, string $assignmentKeyName, array $assignmentKeyValues, string $keyName, int $keyValue): void
    {
        $assigned = $this->database->table($pairingTableName)->where($keyName . ' = ?', $keyValue)->fetchPairs('id', $assignmentKeyName);
        foreach ($assigned as $id => $value) {
            if (!in_array($value, $assignmentKeyValues)) {
                $this->deleteRecord($id, $pairingTableName);
            }
        }
        foreach ($assignmentKeyValues as $value) {
            if (!in_array($value, $assigned)) {
                $this->assign($pairingTableName, [$assignmentKeyName => $value, $keyName => $keyValue]);
            }
        }
    }

    /**
     * Toggles assignment in pairing table
     * @param string $pairingTableName
     * @param string $assignmentKeyName
     * @param mixed $assignmentKeyValue
     * @param string $keyName
     * @param mixed $keyValue
     * @return bool
     */
    public function toggleAssignment(string $pairingTableName, string $assignmentKeyName, $assignmentKeyValue, string $keyName, $keyValue) : bool
    {
        if ($this->database->table($pairingTableName)->where($assignmentKeyName . '= ? AND ' . $keyName . ' = ?', $assignmentKeyValue, $keyValue)->count('*') > 0) {
            return $this->unassign($pairingTableName, [$assignmentKeyName => $assignmentKeyValue, $keyName => $keyValue]);
        } else {
            return $this->assign($pairingTableName, [$assignmentKeyName => $assignmentKeyValue, $keyName => $keyValue]);
        }
    }

}
